package uestc.dm.GILPA_Energy;

import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

public class CostGenerationTechniques
{
	/**
	 * Cost function = (1-alpha_i)(1-alpha_j), only allow a maximal of two labels to be transferred each time.
	 * @param allLabelsWithCost
	 * @param node, the receiver of labels
	 * @param neighbor, the sender of labels
	 */
	public static void maximalTwoLabelsCost(List<LabelWithCost> allLabelsWithCost, Node node, Node neighbor)
	{
		List<Label> allLabels=node.getAllLabels();
		List<Label> neighborAllLabels=neighbor.getAllLabels();
		for(Label neighborLabel:neighborAllLabels)
		{
			boolean isMutualCommunity=false;
			double cost = 0;
			for(Label label:allLabels)
			{
				if(neighborLabel==label)
				{
					cost=(1-label.getCoefficient())*(1-neighborLabel.getCoefficient());
					isMutualCommunity=true;
				}
			}
			if(!isMutualCommunity)
			{
				cost=1-neighborLabel.getCoefficient();
			}
			allLabelsWithCost.add(new LabelWithCost(neighborLabel, cost));
		}
	}
	
	/**
	 * Cost function = |alpha_j-alpha_i|(1-alpha_i)+1/s
	 * @param allLabelsWithCost
	 * @param node, the receiver of labels
	 * @param neighbor, the sender of labels
	 */
	public static void absoluteDifferenceCost(List<LabelWithCost> allLabelsWithCost, Node node, Node neighbor)
	{
		List<Label> allLabels=node.getAllLabels();
		List<Label> neighborAllLabels=neighbor.getAllLabels();
		for(Label neighborLabel:neighborAllLabels)
		{
			boolean isMutualCommunity=false;
			double cost = 1.0/neighborAllLabels.size();
			for(Label label:allLabels)
			{
				if(neighborLabel==label)
				{
					if(label.getCoefficient()>neighborLabel.getCoefficient())
					{
						cost=cost+(label.getCoefficient()-neighborLabel.getCoefficient())*(1.0-neighborLabel.getCoefficient());
					}
					else
					{
						cost=cost+(neighborLabel.getCoefficient()-label.getCoefficient())*(1.0-neighborLabel.getCoefficient());
					}
					isMutualCommunity=true;
				}
			}
			if(!isMutualCommunity)
			{
				cost=cost+(1-neighborLabel.getCoefficient())*neighborLabel.getCoefficient();
			}
			allLabelsWithCost.add(new LabelWithCost(neighborLabel, cost));
		}
	}
	
	/**
	 * Cost function = (1-alpha_i)(alpha_j-alpha_i)^2, favors more labels to be transferred than absolute difference cost.
	 * @param allLabelsWithCost
	 * @param node, the receiver of labels
	 * @param neighbor, the sender of labels
	 */
	public static void squareDifferenceCost(List<LabelWithCost> allLabelsWithCost, Node node, Node neighbor)
	{
		List<Label> allLabels=node.getAllLabels();
		List<Label> neighborAllLabels=neighbor.getAllLabels();
		for(Label neighborLabel:neighborAllLabels)
		{
			boolean isMutualCommunity=false;
			double cost = 0;
			for(Label label:allLabels)
			{
				if(neighborLabel==label)
				{
					cost=(1-neighborLabel.getCoefficient())*(neighborLabel.getCoefficient()-label.getCoefficient())*(neighborLabel.getCoefficient()-label.getCoefficient());
					isMutualCommunity=true;
				}
			}
			if(!isMutualCommunity)
			{
				cost=(1-neighborLabel.getCoefficient())*neighborLabel.getCoefficient();
			}
			allLabelsWithCost.add(new LabelWithCost(neighborLabel, cost));
		}
	}
	
	/**
	 * Cost function = |alpha_j-alpha_i|+(1-alpha_i)
	 * @param allLabelsWithCost
	 * @param node, the receiver of labels
	 * @param neighbor, the sender of labels
	 */
	public static void absoluteDifferenceAugmentCost(List<LabelWithCost> allLabelsWithCost, Node node, Node neighbor)
	{
		List<Label> allLabels=node.getAllLabels();
		List<Label> neighborAllLabels=neighbor.getAllLabels();
		for(Label neighborLabel:neighborAllLabels)
		{
			boolean isMutualCommunity=false;
			double cost = 0;
			for(Label label:allLabels)
			{
				if(neighborLabel==label)
				{
					if(label.getCoefficient()>neighborLabel.getCoefficient())
					{
						cost=label.getCoefficient()-neighborLabel.getCoefficient()+1.0-neighborLabel.getCoefficient();
					}
					else
					{
						cost=neighborLabel.getCoefficient()-label.getCoefficient()+1.0-neighborLabel.getCoefficient();
					}
					isMutualCommunity=true;
				}
			}
			if(!isMutualCommunity)
			{
				cost=1-neighborLabel.getCoefficient()+neighborLabel.getCoefficient();
			}
			allLabelsWithCost.add(new LabelWithCost(neighborLabel, cost));
		}
	}
	
	/**
	 * Cost function = |alpha_j-alpha_i|, only send the minimal label.
	 * @param allLabelsWithCost
	 * @param node, the receiver of labels
	 * @param neighbor, the sender of labels
	 */
	public static void minimalAbsoluteDifferenceCost(List<LabelWithCost> allLabelsWithCost, Node node, Node neighbor)
	{
		List<Label> allLabels=node.getAllLabels();
		List<Label> neighborAllLabels=neighbor.getAllLabels();
		for(Label neighborLabel:neighborAllLabels)
		{
			boolean isMutualCommunity=false;
			double cost = 0;
			for(Label label:allLabels)
			{
				if(neighborLabel==label)
				{
					if(label.getCoefficient()>neighborLabel.getCoefficient())
					{
						cost=label.getCoefficient()-neighborLabel.getCoefficient();
					}
					else
					{
						cost=neighborLabel.getCoefficient()-label.getCoefficient();
					}
					isMutualCommunity=true;
				}
			}
			if(!isMutualCommunity)
			{
				cost=neighborLabel.getCoefficient();
			}
			allLabelsWithCost.add(new LabelWithCost(neighborLabel, cost));
		}
		//Delete all the labels except that with minimal cost.
		Collections.sort(allLabelsWithCost);
		Iterator<LabelWithCost> it=allLabelsWithCost.iterator();
		it.next();
		while (it.hasNext())
		{
			it.next();
			it.remove();
		}
	}

}
